--[[
    DCS Mission Script: Automated Cruise Missile Strike v1
    Description:
        - After placing a map marker labeled "TARGET", automatically scans for enemy units within 3km (ground + static).
        - Multiple ships fire cruise missiles and temporarily switch Rules of Engagement (ROE) to "Fire".
        - ROE is restored only after all missiles are launched, and a message is displayed.
        - Displays missile launch summary per ship.
        - Audio alert support included.

--]]

local maxMissilesPerShip = 22
local markerPrefix = "TARGET"
local blueMissiles = { "BGM_109" }
local redMissiles = { "3M-54" }
local engagementRadius = 3000

local shipData = {
    [coalition.side.BLUE] = {},
    [coalition.side.RED] = {}
}

local function detectCruiseMissileShips()
    for coalitionID in pairs({ [coalition.side.BLUE]=true, [coalition.side.RED]=true }) do
        local detectedShips = {}
        local allGroups = coalition.getGroups(coalitionID, Group.Category.NAVAL)
        for _, group in ipairs(allGroups) do
            for _, unit in ipairs(group:getUnits()) do
                if unit and unit:isExist() and unit:isActive() then
                    local unitAmmo = unit:getAmmo()
                    if unitAmmo then
                        local missileTypes = (coalitionID == coalition.side.BLUE) and blueMissiles or redMissiles
                        for _, ammo in ipairs(unitAmmo) do
                            if ammo.desc then
                                for _, missileType in ipairs(missileTypes) do
                                    if ammo.desc.typeName and ammo.desc.typeName:find(missileType) then
                                        detectedShips[unit:getName()] = {
                                            name = unit:getName(),
                                            inventory = maxMissilesPerShip,
                                            unit = unit,
                                            lastROE = nil,
                                            roeChanged = false
                                        }
                                        break
                                    end
                                end
                            end
                        end
                    end
                end
            end
        end
        shipData[coalitionID] = detectedShips
    end
end

local function getDistance2D(p1, p2)
    local dx = p1.x - p2.x
    local dz = p1.z - p2.z
    return math.sqrt(dx * dx + dz * dz)
end

local function fireMissile(ship, targetPos)
    local controller = ship.unit:getController()
    if controller and ship.inventory > 0 then
        local task = {
            id = 'FireAtPoint',
            params = {
                x = targetPos.x,
                y = targetPos.z,
                radius = 100,
                expendQty = 1,
                expendQtyEnabled = true
            }
        }
        controller:pushTask(task)
        ship.inventory = ship.inventory - 1
        return true
    end
    return false
end

local function setROETemporary(ship)
    local controller = ship.unit:getController()
    if controller and not ship.roeChanged then
        if controller.getOption then ship.lastROE = controller:getOption(0) end
        if controller.setOption then controller:setOption(0, 2) end -- 2 = Fire
        ship.roeChanged = true
    end
end

local function restoreROE(ship)
    local controller = ship.unit:getController()
    if controller and ship.roeChanged and ship.lastROE ~= nil then
        if controller.setOption then controller:setOption(0, ship.lastROE) end
        ship.roeChanged = false
    end
end

local function handleMapMarker(event)
    local rawText = event.text or ""
    local cleanedText = string.gsub(string.upper(rawText), "%s", "")

    if event.id ~= world.event.S_EVENT_MARK_CHANGE then return end
    if cleanedText ~= markerPrefix then return end

    local coalitionID = event.coalition
    local enemyCoalition = (coalitionID == coalition.side.BLUE) and coalition.side.RED or coalition.side.BLUE
    local targetCenter = event.pos

    local targets = {}

    for _, group in ipairs(coalition.getGroups(enemyCoalition, Group.Category.GROUND)) do
        for _, unit in ipairs(group:getUnits()) do
            if unit:isExist() then
                local pos = unit:getPoint()
                if getDistance2D(pos, targetCenter) <= engagementRadius then
                    table.insert(targets, pos)
                end
            end
        end
    end

    for _, staticObj in ipairs(coalition.getStaticObjects(enemyCoalition)) do
        if staticObj:isExist() then
            local pos = staticObj:getPoint()
            if getDistance2D(pos, targetCenter) <= engagementRadius then
                table.insert(targets, pos)
            end
        end
    end

    trigger.action.outText(string.format("[Strike] Marker '%s': Detected %d enemy targets", rawText, #targets), 10)
    if #targets == 0 then return end

    -- 播放音效提示

    local ships = {}
    for _, ship in pairs(shipData[coalitionID]) do
        if ship.unit and ship.unit:isExist() and ship.inventory > 0 then
            ship.distance = getDistance2D(ship.unit:getPoint(), targetCenter)
            table.insert(ships, ship)
        end
    end

    if #ships == 0 then
        trigger.action.outText("[Strike] No available ships", 10)
return
    end

    table.sort(ships, function(a, b) return a.distance < b.distance end)

    -- Switch ROE to Fire before firing
    for _, ship in ipairs(ships) do
        setROETemporary(ship)
    end
    trigger.action.outText("[ROE] Fleet engagement rule set to FIRE", 5)

    local shipIndex = 1
    local summary = {}
    for _, target in ipairs(targets) do
        for i = 1, #ships do
            local ship = ships[shipIndex]
            if ship.inventory > 0 then
                if fireMissile(ship, target) then
                    summary[ship.name] = (summary[ship.name] or 0) + 1
                end
                shipIndex = shipIndex % #ships + 1
                break
            end
            shipIndex = shipIndex % #ships + 1
        end
    end

    local report = "[TARGET] Launch Summary:\n"
    for name, count in pairs(summary) do
        report = report .. string.format("  %s: %d missiles\n", name, count)
    end
    trigger.action.outText(report, 7)

    -- Restore ROE after 3 seconds
    timer.scheduleFunction(function()
        for _, ship in ipairs(ships) do
            restoreROE(ship)
        end
        trigger.action.outText("[ROE] Fleet engagement rule restored", 5)
    end, nil, timer.getTime() + 3)

    if event.idx then
        trigger.action.removeMark(event.idx)
    end
end

detectCruiseMissileShips()

world.addEventHandler({
    onEvent = function(self, event)
        if event.id == world.event.S_EVENT_MARK_CHANGE then
            handleMapMarker(event)
        end
    end
})

trigger.action.outText("Automated Cruise Missile Strike v1", 5)